function [] = QSpline_Plot_Contour( dt, Fsample, n)
%Plot a Q_Spline over a given triangulation
%   First input the triangulation from delaunaytri.  The Fsample should be
%   the output from Q_Fsample; hence, it is the values of Qspline at triangle
%   vertices and midpoints.  n is the number of sample points taken per
%   triangle edge {controls how many samples in the interior also. Total
%   points in \sum^{n}_{i=1} i = n(n+1)/2 }

numtri = size(dt.Triangulation,1);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Interpolate the different splines indexed to the different triangles.
%Basically is second part of QSpline_Eval. In context of code, more
%efficient for time and computation if do this instead of running
%QSpline_Eval (means I'd map Std element 

%The rows store the spline coefficients on {1,x,y,xy,x^2,y^2} basis for
%Point i
C = zeros(numtri, 6);

for i=1:numtri
    N = [ Fsample(i,:,1)' Fsample(i,:,2)' ];
    p = Fsample(i,:,3)' ;
    C(i,:) = Q_Terp(N, p);
end
clear N p
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%We need to pick the test points which we'll sample the spline at for
%use in Matlab's plot function.  The idea here is that we'll seed the std
%element to some precision, then map these points into each triangle via
%Std_to_Tri. 'n' is the key parameter, being the number of points taken
%along e1, and one less each line above until the vertex e2 at which there
%is just the 1.

%Grid the standard element
s = 0:1/(n-1):1;
t = 0:1/(n-1):1;
[S T] = meshgrid(s,t); %This gives us grids for a square but need to only 
                       %be our triangle!
                       
S = 1 - S;          %For those vertices of square grid not in <0,e1,e2> triangle
S = triu(S);        %pad those with a repeat of the origin. This ensures
T = triu(T);        %we only mesh the triangle. NOTE: The upper triangular part
                    %with the diagonal is carrying actual info on triangle.
                    %The STRICTLY lower part is padded as 0 in both x and y

%We now need to map these standard points into each triangle within our
%actual triangulation.

X = zeros(n*numtri,n); %To each triangle is n x n grid that is 
Y = zeros(n*numtri,n); %'padded' with the repeat vertex to make triangular
Z = zeros(n*numtri,n); %Where we'll store the Z-values
I = zeros(n,n)+1; %We'll use to make sure true padded and no numerical roundoff error
I = triu(I); %Is a strictly upper triangle matrix with 1's.

h = waitbar(0,'How Many Triangles Have Been Evaluated');
for i=1:numtri  %Indexes which triangle we're at.
    Triangle = zeros(3,2);
    Triangle(:,1) = arrayfun(@(l) dt.X(l,1), dt.Triangulation(i,:)'); %Extract triangle vertices: x-coordinates
    Triangle(:,2) = arrayfun(@(l) dt.X(l,2), dt.Triangulation(i,:)'); %Extract triangle vertices: y-coordinates
    
    X((i-1)*n+1:i*n,:) = arrayfun(@(u,v) Std_to_Tri(Triangle,[[u v]])*[[1 0]]', S, T); %Map in the points: x-coordinates
    Y((i-1)*n+1:i*n,:) = arrayfun(@(u,v) Std_to_Tri(Triangle,[[u v]])*[[0 1]]', S, T); %Map in the points: y-coordinates
    
    B = triu(X((i-1)*n+1:i*n,:)) - diag(diag(X((i-1)*n+1:i*n,:))); %Instead of padding by a single point which made surf look buggy,
    X((i-1)*n+1:i*n,:) = X((i-1)*n+1:i*n,:).*I+B';                 %We double all off diagonal points, we'll see if better
    
    B = triu(Y((i-1)*n+1:i*n,:)) - diag(diag(Y((i-1)*n+1:i*n,:))); % 
    Y((i-1)*n+1:i*n,:) = Y((i-1)*n+1:i*n,:).*I+B';
    
    %X((i-1)*n+1:i*n,:) = triu(X((i-1)*n+1:i*n,:)) + X((i-1)*n+1,1).*(I); %Ensures strictly lower part of triangle is padded with the (1,1) entry
    %Y((i-1)*n+1:i*n,:) = triu(Y((i-1)*n+1:i*n,:)) + Y((i-1)*n+1,1).*(I); %Same but for Y.  (These two lines are just a fail-safe, should be this already.)

    %With the points in the ith triangle known, we can go ahead and compute
    %the spline values
    
    Z((i-1)*n+1:i*n,:) = arrayfun(@(u,v) C(i,1)+C(i,2)*u+C(i,3)*v+C(i,4)*u*v+C(i,5)*u^2+C(i,6)*v^2, X((i-1)*n+1:i*n,:), Y((i-1)*n+1:i*n,:)); %Evaluate the spline using C coefficients.
    B = triu(Z((i-1)*n+1:i*n,:)) - diag(diag(Z((i-1)*n+1:i*n,:))); %again how we duplicate to make surf behave better
    Z((i-1)*n+1:i*n,:) = Z((i-1)*n+1:i*n,:).*I+B';
    
    %Z((i-1)*n+1:i*n,:) = triu(Z((i-1)*n+1:i*n,:)) + Z((i-1)*n+1,1).*(I); %Again fail-safe to make sure no roundoff error.
    
    waitbar(i/numtri,h)

end
close(h)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Render the spline in a new figure diagram
figure;
hold
for i=1:numtri
surfc(X((i-1)*n+1:i*n,:),Y((i-1)*n+1:i*n,:),Z((i-1)*n+1:i*n,:),...
    'FaceColor','interp',...
    'EdgeColor','none',...
    'FaceLighting','gouraud')%,'light') %We plot each individual triangle so that Matlab doesn't try to connect what shouldnt connect
%colormap hsv
end
hold

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Helpful Matlab Documentation: http://www.mathworks.com/help/matlab/visualize/representing-a-matrix-as-a-surface.html

end

